﻿"""Implementation of an orientation transport.
"""

import vizmat

import transportation


class Orientation(transportation.AccelerationTransport):
	"""Orientation transports can be used to look around a scene, or to have
	an automated rotation of a view.
	"""
	
	def __init__(self,
				autoOrientationSpeedIncrement=2.0,
				acceleration=4.0, # in meters per second per second, lower accelerations can be obtained by using a smaller mag on the input, e.g. pressing the joystick lower
				maxSpeed=10.44, # in meters per second, as a reference 1.4m/s is a typical walking speed, 10.44 is a very fast run
				rotationAcceleration=90.0, # in degrees per second per second
				maxRotationSpeed=120.0, # in degrees per second
				autoBreakingDragCoef=0.1, # determines how quickly the walking transport will stop 
				dragCoef=0.0001,
				rotationAutoBreakingDragCoef=0.3, # determines how quickly the walking transport will stop 
				rotationDragCoef=0.0001, # normal drag coef
				rotationAutoBreakingTimeout=0, # how long before rotational auto breaking is enabled
				**kwargs):
		
		super(Orientation, self).__init__(acceleration=acceleration,
									maxSpeed=maxSpeed,
									rotationAcceleration=rotationAcceleration,
									maxRotationSpeed=maxRotationSpeed,
									autoBreakingDragCoef=autoBreakingDragCoef,
									dragCoef=dragCoef,
									rotationAutoBreakingDragCoef=rotationAutoBreakingDragCoef,
									rotationDragCoef=rotationDragCoef,
									**kwargs)
		
		self.LOCK_INCREASE_orientation_LEFT = 1
		self.LOCK_INCREASE_orientation_RIGHT = 2
		
		self._orientationSpeed = [0, 0, 0]
		self._autoOrientationSpeedIncrement = autoOrientationSpeedIncrement
		self._rotationAutoBreakingTimeout = rotationAutoBreakingTimeout
		
		self._autoBreakingTimeoutCounter = 0
		self._rotationAutoBreakingTimeoutCounter = [0, 0, 0]
		
		self.setUpdateFunction(lambda transport: None)
		
		mat = self.getMatrix()
		self._yaw, self._pitch, self._roll = mat.getEuler()

	
	def getAccelerationRotationMatrix(self):
		mat = vizmat.Transform()
		mat.setQuat(self.getQuat())
		return mat
	
	def left(self, mag=1):
		"""Turns left"""
		self._Ar[0] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def right(self, mag=1):
		"""Turns right"""
		self._Ar[0] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def rollLeft(self, mag=1):
		"""Rolls left"""
		self._Ar[2] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def rollRight(self, mag=1):
		"""Rolls right"""
		self._Ar[2] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def up(self, mag=1):
		"""Looks up"""
		self._Ar[1] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def down(self, mag=1):
		"""Looks down"""
		self._Ar[1] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def increaseOrientationLeft(self, mag=1):
		"""Increase the orientation speed, turning left, in the case of
		auto turning.
		"""
		if not self._frameLocked&self.LOCK_INCREASE_orientation_LEFT:
			self._orientationSpeed[0] += self._autoOrientationSpeedIncrement
		self._lockRequested |= self.LOCK_INCREASE_orientation_LEFT
	
	def increaseOrientationRight(self, mag=1):
		"""Increase the orientation speed, turning right, in the case of
		auto turning.
		"""
		if not self._frameLocked&self.LOCK_INCREASE_orientation_RIGHT:
			self._orientationSpeed[0] -= self._autoOrientationSpeedIncrement
		self._lockRequested |= self.LOCK_INCREASE_orientation_RIGHT
	
	def setOrientationSpeed(self, orientationSpeed):
		"""Sets the orientation speed, left/right, in the case of
		auto turning.
		"""
		self._orientationSpeed = orientationSpeed[:]
	
	def finalize(self):
		"""Function which executes the quequed functions such as
		moveForward and moveBack basing them off the sample orientation
		from the tracker. Should be called regularly either by a timer
		or ideally at every frame.
		"""
		self._frameLocked = self._lockRequested
		self._lockRequested = 0
		
		self._updateTime()
		if self._dt == 0:
			self._dt = 0.0001
		idt = min(60.0, 1.0/self._dt)
		
		# if necessary normalize the acceleration
		mag = self._Ap.length()
		if mag > 1.0:
			self._Ap = self._Ap / mag
		# .. and for rotation
		mag = self._Ar.length()
		if mag > 1.0:
			self._Ar = self._Ar / mag
		
		# scale acceleration (right now no units just 0-1 range magnitude vector)
		self._Ap *= self._acceleration
		# .. and for rotation
		self._Ar *= self._rotationAcceleration
		
		rotMat = self.getAccelerationRotationMatrix()
		
		invMat = rotMat.inverse()
		
		# update velocity for position/zoom
		breaking = self._Vp[2] * self._autoBreakingDragCoef
		localVp = invMat.preMultVec(self._Vp)
		if self._Ap[2] != 0 and (self._Ap[2]*localVp[2] > 0):
			breaking = 0
		
		drag = self._Vp[2] * self._dragCoef
		self._Vp[2] += (self._Ap[2] - drag - breaking)*self._dt
		velMag = abs(self._Vp[2])
		if velMag > self._maxSpeed:
			self._Vp[2] = (self._Vp[2] / velMag) * self._maxSpeed
		
		# .. and for rotation
		breakingVec = self._Vr * self._rotationAutoBreakingDragCoef * idt
		for i in range(0, 3):
			if self._Ar[i] != 0:
				breakingVec[i] = 0
			if breakingVec[i]:
				if self._rotationAutoBreakingTimeoutCounter[i] < self._rotationAutoBreakingTimeout:
					breakingVec[i] = 0# cancel breaking
				self._rotationAutoBreakingTimeoutCounter[i] += self._dt
			else:
				self._rotationAutoBreakingTimeoutCounter[i] = 0
		
		drag = self._Vr * self._rotationDragCoef
		self._Vr[0] += (self._Ar[0] - drag[0] - breakingVec[0]) * self._dt
		self._Vr[1] += (self._Ar[1] - drag[1] - breakingVec[1]) * self._dt
		self._Vr[2] += (self._Ar[2] - drag[2] - breakingVec[2]) * self._dt
		velMag = self._Vr.length()
		if velMag > self._maxRotationSpeed:
			self._Vr = (self._Vr / velMag) * self._maxRotationSpeed
		
		self._yaw += self._Vr[0]*self._dt + self._orientationSpeed[0]*self._dt
		#pitch = min(89.5, max(-89.5, pitch + self._Vr[1]*self._dt + self._orientationSpeed[1]*self._dt))
		self._pitch += self._Vr[1]*self._dt + self._orientationSpeed[1]*self._dt
		self._roll += self._Vr[2]*self._dt + self._orientationSpeed[2]*self._dt
		
		self.setEuler([self._yaw, self._pitch, self._roll])
		
		self._Ap[0] = 0
		self._Ap[1] = 0
		self._Ap[2] = 0
		self._Ar[0] = 0
		self._Ar[1] = 0
		self._Ar[2] = 0


if __name__ == "__main__":
	import viz
	import vizact
	viz.go()
	
	# load a model
	piazza = viz.add('piazza.osgb')
	
	transportRepresentation = viz.add("beachball.osgb")
	orientation = Orientation(autoOrientationSpeedIncrement=2.0,
				acceleration=4.0, # in meters per second per second, lower accelerations can be obtained by using a smaller mag on the input, e.g. pressing the joystick lower
				maxSpeed=10.44, # in meters per second, as a reference 1.4m/s is a typical walking speed, 10.44 is a very fast run
				rotationAcceleration=90.0, # in degrees per second per second
				maxRotationSpeed=120.0, # in degrees per second
				autoBreakingDragCoef=0.1, # determines how quickly the walking transport will stop 
				dragCoef=0.0001,
				rotationAutoBreakingDragCoef=0.2, # determines how quickly the walking transport will stop 
				rotationDragCoef=0.0001,
				rotationAutoBreakingTimeout=0.1) # how long before rotational auto breaking is enabled
	
	vizact.onkeydown(viz.KEY_UP, orientation.up)
	vizact.onkeydown(viz.KEY_DOWN, orientation.down)
	vizact.onkeydown(viz.KEY_LEFT, orientation.left)
	vizact.onkeydown(viz.KEY_RIGHT, orientation.right)
	vizact.onkeydown('q', orientation.rollLeft)
	vizact.onkeydown('e', orientation.rollRight)
	
	base = viz.addGroup()
	box = viz.add("arrow.wrl")
	box.setPosition(0, 0, 1.5)
	box.setScale([0.15]*3)
	box.setParent(base)
	
	vizact.onkeydown('5', orientation.setNode, base)
	
	group = viz.addGroup()
	group.setParent(orientation)
	group.setEuler([0, 0, 0])
	link = viz.link(group, viz.MainView)
	link.setSrcFlag(viz.ABS_GLOBAL)
#	link = viz.link(orientation, viz.MainView)

